#ifndef _APPGRAPHICS_HEADERS
#define _APPGRAPHICS_HEADERS
#include <windows.h>        
#include <limits.h>         
#include <stdbool.h>        

#include "agconstants.h"

/* ---------------------------------------------------------------------------
 *                              Structures
 * ---------------------------------------------------------------------------
 */
 
/* This structure records information about the last call to arc.  It is used
 * by getarccoords to get the location of the endpoints of the arc.
 */
typedef struct
{
    int x, y;                   // Center point of the arc
    int xstart, ystart;         // The starting position of the arc
    int xend, yend;             // The ending position of the arc.
} arccoordstype;


/* This structure defines the fill style for the current window.  Pattern is
 * one of the system patterns such as SOLID_FILL.  Color is the color to
 * fill with
 */
typedef struct 
{
    int pattern;                // Current fill pattern
    int color;                  // Current fill color
    int bkmode;
} fillsettingstype;


/* This structure records information about the current line style.
 * linestyle is one of the line styles such as SOLID_LINE, upattern is a
 * 16-bit pattern for user defined lines, and thickness is the width of the
 * line in pixels.
 */
typedef struct 
{
    int linestyle;              // Current line style
    unsigned short upattern;    // 16-bit user line pattern
    int thickness;              // Width of the line in pixels
    int bkmode;
} linesettingstype;


/* This structure records information about the text settings. */
typedef struct 
{
    int font;                   // The font in use
    int direction;              // Text direction
    int charsize;               // Character size
    int horiz;                  // Horizontal text justification
    int vert;                   // Vertical text justification
    int bkmode;                 // Background mode
} textsettingstype;


/* This structure records information about the viewport */
typedef struct 
{
    int left, top,              // Viewport bounding box
        right, bottom;
    int clip;                   // Whether to clip image to viewport
} viewporttype;

/* ---------------------------------------------------------------------------
 *                     Legacy BGI Compatibility
 * ---------------------------------------------------------------------------
 */

#ifdef BGI_COMPATIBLE
#define LEGACY
#endif

#ifdef LEGACY
#include "aglegacy.h"
#endif

/* ---------------------------------------------------------------------------
 *                          Callback Entries
 * ---------------------------------------------------------------------------
 */
typedef void (*MouseHandler)(int, int);

typedef void (*IntegerCallback)(int);

typedef int (*IntegerReturnedCallback)();

typedef void (*SimpleCallback)();

typedef SimpleCallback MenuHandler;
typedef SimpleCallback ButtonHandler;
typedef IntegerCallback ScrollHandler;
typedef SimpleCallback ListHandler;
typedef SimpleCallback WindowCloseHandler;
typedef IntegerReturnedCallback SystemCloseHandler;


/* ---------------------------------------------------------------------------
 *                          API Entries
 * ---------------------------------------------------------------------------
 */
    
#ifdef __cplusplus
extern "C" {
#endif

/* Drawing Functions */
void arc( int x, int y, int stangle, int endangle, int radius );
void bar( int left, int top, int right, int bottom );
void bar3d( int left, int top, int right, int bottom, int depth, int topflag );
void circle( int x, int y, int radius );
void cleardevice( );
void drawpoly(int n_points, int* points);
void ellipse( int x, int y, int stangle, int endangle, int xradius, int yradius );
void fillellipse( int x, int y, int xradius, int yradius );
void fillpoly(int n_points, int* points);
void floodfill( int x, int y, int border );
void line( int x1, int y1, int x2, int y2 );
void linerel( int dx, int dy );
void lineto( int x, int y );
void pieslice( int x, int y, int stangle, int endangle, int radius );
void putpixel( int x, int y, int color );
void rectangle( int left, int top, int right, int bottom );
void sector( int x, int y, int stangle, int endangle, int xradius, int yradius );
void bezier(int x1, int y1, int x2, int y2, int cx1, int cy1, int cx2, int cy2);

/* Color Functions */
int getdisplaycolor( int color );
int getbkcolor( );
int getcolor( );
void setbkcolor( int color );
void setcolor( int color );
int creatergb(int r, int g, int b); 
int getredvalue( int color );
int getgreenvalue( int color );
int getbluevalue( int color );

/* Fill/Line Settings Functions */
void getfillpattern( char *pattern );
void getfillsettings( fillsettingstype *fillinfo );
void setfillsettings( fillsettingstype fillinfo );
void getlinesettings( linesettingstype *lineinfo );
void setlinesettings( linesettingstype lineinfo );
void setfillpattern( char *upattern, int color );
void setfillstyle( int pattern, int color );
void setlinestyle( int linestyle, unsigned upattern, int thickness );

void delay( int msec );

/* Window Routines */
int getmaxheight( );
int getmaxwidth( );
int getmaxx( );
int getmaxy( );
int getwindowheight( );
int getwindowwidth( );
void enableresize( SimpleCallback cb );
void disableresize( );
void setwindowtitle( char *title );
void setwindowlargeicon( int icon );
void setwindowsmallicon( int icon );

/* Query Functions */
int getpixel( int x, int y );
int getx( );
int gety( );
void getarccoords( arccoordstype *arccoords );
int getsignallingwindow( );

/* Positioning Functions */
void moverel( int dx, int dy );
void moveto( int x, int y );

/* Screen refresh */
void refreshrect(int left, int top, int right, int bottom);
void refreshall( );    
void setrefreshing(bool value);
bool getrefreshing( );
void setwritemode( int mode );

/* Viewports */
void clearviewport( );
void getviewsettings( viewporttype *viewport );
void setviewport( int left, int top, int right, int bottom, int clip );
void resetviewport();
void updateviewport();

/* Window Creation / Graphics Manipulation */
void closewindow( int wid );
void getaspectratio( int *xasp, int *yasp );
void graphdefaults( );
char *grapherrormsg( int errorcode );
int graphresult( );
int initwindow( int width, 
                int height, 
                const char* title, 
                int left, 
                int top, 
                bool dbflag, 
                bool closeflag );
void setaspectratio( int xasp, int yasp );
void setdialoghotkeys( bool enable );

/* DPI Routines */
void setapplicationdpiaware( );
int getdpi();
int scaledpi(int desired);

/* User Interaction */
int getch( );
int kbhit( );

/* User-Controlled Window Function  */
int getcurrentwindow( );
void setcurrentwindow( int window );
bool isvalidwindow( int window );
void setwindowclosecallback( WindowCloseHandler cb );
void setwindowsystemclosecallback( SystemCloseHandler cb );
    
/* Double -buffering support */
int getactivepage( );
int getvisualpage( );
void setactivepage( int page );
void setvisualpage( int page );
void swapbuffers( );

/* Image Functions */
long imagesize(long width, long height);
void *mallocimage(long width, long height); /* Release with "free" */
void getimage( int left, int top, int right, int bottom, void *bitmap );
void putimage( int left, int top, void *bitmap, int op );
void scaleimage( void *bitmapSrc, void *bitmapDest, int width, int height ); /* bitmapDest must already be appropriately allocated */
void *loadimage(const char *filename, int *width, int *height); /* Release with "free" */
void printimage(
    const char* title,
    int left, int top, int right, int bottom,
    bool active
    );
void readimagefile(
    const char* filename,
    int left, int top, int right, int bottom
    );
void writeimagefile(
    const char* filename,
    int left, int top, int right, int bottom,
    bool active
    );

/* Text Functions */
void useansitext();
void gettextsettings(textsettingstype *texttypeinfo);
void settextsettings(textsettingstype texttypeinfo);
void outtext(const TCHAR *textstring);
void outtextxy(int x, int y, const TCHAR *textstring);
void settextxy(int x, int y);
void settextjustify(int horiz, int vert);
void settextstyle(int font, int direction, int charsize);
void setusercharsize(int multx, int divx, int multy, int divy);
int textheight(const TCHAR *textstring);
int textwidth(const TCHAR *textstring);
void sendtoclipboard(const TCHAR *textstring);
size_t getclipboardtextlength();
size_t getfromclipboard(TCHAR *buffer, size_t buflen);
    
/* Mouse Functions */
void clearmouseclick( int kind );
void getmouseclick( int kind, int *x, int *y );
bool ismouseclick( int kind );
int mousex( );
int mousey( );
void registermousehandler( int kind, MouseHandler cb );
void setmousequeuestatus( int kind, bool status);

/* Palette Functions */
int systemcolor(int index);

/* Color Macros */
#define RED_VALUE(v)          ((int)GetRValue( v ))
#define GREEN_VALUE(v)        ((int)GetGValue( v ))
#define BLUE_VALUE(v)         ((int)GetBValue( v ))

/* The Window Menu  API  */
int addmenu(const TCHAR *title, int parent);
void removemenu(int i);
int addmenuitem(const TCHAR *title, int parent, MenuHandler cb);
int addmenuseparator(int parent);
void removemenuitem(int id);
int getmenuhandlerid();

/* For idling on the main thread */
void startidle(long waitTimeMS);
void stopidle();
void loop();

/* Mutexes & Threads */
int createmutex();
bool lockmutex(int id, long wait);
void unlockmutex(int id);

/* Window controls */
int getcallbackcontrolid();

int createbutton(int x, int y, int w, int h, const TCHAR *label, ButtonHandler cb);
void removebutton(int id);
void setbuttonposition(int index, int left, int top, int width, int height);
bool getbuttonposition(int index, int *left, int *top, int *width, int *height);
#define getcallbackbuttonid() getcallbackcontrolid()

int createtextbox(int x, int y, int w, int h);
int createmultilinetextbox(int x, int y, int w, int h, bool line_wrap);
void removetextbox(int index);
int gettextboxlength(int index);
int gettextboxcontents(int index, char *buffer, int buflength);
void settextboxcontents(int index, const char *text);
int gettextboxinteger(int index);
void settextboxinteger(int index, int value);
double gettextboxdouble(int index);
void settextboxdouble(int index, double value);
void settextboxposition(int index, int left, int top, int width, int height);
bool gettextboxposition(int index, int *left, int *top, int *width, int *height);
void settextboxentercallback(int index, SimpleCallback cb);
int gettextboxselection(int index, char *buffer, int buflength);
int gettextboxselectionlength(int index);
void inserttextboxtext(int index, const char *text);
void deletetextboxselection(int index);
int gettextboxcursorposition(int index);
int gettextboxlinefromposition(int index, int position);
int gettextboxpositionfromline(int index, int line);
void settextboxcursorposition(int index, int position);
void textboxselect(int index, int charcount);
void settextboxfocused(int index);
void settextboxcue(int index, const char *text);
void settextboxvisible(int index, bool visible);
#define getcallbacktextboxid() getcallbackcontrolid()

/* Scroll bars */
int createscrollbar(int x, int y, int w, int h, int direction, ScrollHandler cb);
void removescrollbar(int index);
void setscrollrange(int index, int minPos, int maxPos);
void setscrollpagesize(int index, int pageIncrement);
void setscrollposition(int index, int pos);
int getscrollposition(int index);
void setscrollbarposition(int index, int left, int top, int width, int height);
bool getscrollbarposition(int index, int *left, int *top, int *width, int *height);
void setscrollbarvisible(int index, bool visible);
#define getcallbackscrollbarid() getcallbackcontrolid()

/* Check boxes */
int createcheckbox(int x, int y, int w, int h, const TCHAR *label, SimpleCallback cb);
void removecheckbox(int index);
bool checkboxischecked(int index);
void checkboxsetchecked(int index, bool value);
void setcheckboxposition(int index, int left, int top, int width, int height);
bool getcheckboxposition(int index, int *left, int *top, int *width, int *height);
void setcheckboxvisible(int index, bool visible);
#define getcallbackcheckboxid() getcallbackcontrolid()

/* Radio buttons */
int createradiobutton(int x, int y, int w, int h, const TCHAR *label, SimpleCallback cb);
void removeradiobutton(int index);
bool radiobuttonischecked(int index);
void radiobuttonsetchecked(int index, bool value);
void beginradiogroup();
void setradiobuttonposition(int index, int left, int top, int width, int height);
bool getradiobuttonposition(int index, int *left, int *top, int *width, int *height);
void setradiobuttonvisible(int index, bool visible);
#define getcallbackradiobuttonid() getcallbackcontrolid()

/* List boxes */
int createlistbox(int x, int y, int w, int h, SimpleCallback cb);
void clearlistboxentries(int index);
void removelistbox(int index);
int insertlistboxentry(int index, int position, const char *text);
int removelistboxentry(int index, int position);
int getlistboxlength(int index);
int getlistboxtext(int index, int position, char *text, int buffer_length);
int getlistboxselection(int index);
void setlistboxposition(int index, int left, int top, int width, int height);
bool getlistboxposition(int index, int *left, int *top, int *width, int *height);
void setlistboxselection(int index, int entry);
void setlistboxvisible(int index, bool visible);
#define getcallbacklistboxid() getcallbackcontrolid()

/* Combo boxes */
int createdropdownlist(int x, int y, int w, int h, SimpleCallback cb);
void setdropdownlistposition(int index, int left, int top, int width, int height);
void cleardropdownlistentries(int index);
int insertdropdownlistentry(int index, int position, const char *text);
int removedropdownlistentry(int index, int position);
int getdropdownlistlength(int index);
int getdropdownlisttext(int index, int position, char *text, int buffer_length);
int getdropdownlistselection(int index);
bool getdropdownlistposition(int index, int *left, int *top, int *width, int *height);
void setdropdownlistselection(int index, int entry);
void setdropdownlistvisible(int index, bool visible);
#define getcallbackdropdownlistid() getcallbackcontrolid()

/* Dialogs */
bool dlgopenfile(const char *title, char *filename, int filelength, const char *maskdesc, const char *mask);
bool dlgsavefile(const char *title, char *filename, int filelength, const char *maskdesc, const char *mask);
void dlgmessage(int type, const char *message);
bool dlgyesno(int type, const char *message);
int dlgrequestselection(char *items, const char *title, const char *label);
bool dlgrequesttext(char *text, int text_length, const char *title, const char *label);
bool dlgrequestpassword(char *text, int text_length, const char *title, const char *label);

/* Dangerous, platform specific */
#ifdef __WIN32__
HWND getwin32handle();
#endif

#ifdef __cplusplus
}
#endif

#endif /* _APPGRAPHICS_HEADERS */

